SASM is a stack-oriented programming language that is based on Assembly style syntax.

	init:
		print "Hello World!"
		ret

To run the interpreter, you will need a Java 21 Runtime Environment.

	java -jar sasm.jar helloworld.init

Visit https://slashbinbash.de/sasm.html for more information.

# Language

## Comments

Comments are prefaced by a semi-colon `;` and can appear anywhere in the code:

	;comment
	push A  ;comment

## Data-Types

The standard data-types are:

	Boolean  true, false
	Integer  42
	String   "string"
	Name     /foo
	List     [true, 1, "string", [/foo, bar]]

## Instructions

An instruction consist of the name of the instruction, followed by a list of arguments that are separated by commas `,`.

	[instruction] [arg0],[arg1],...,[argN]

The language has instructions for doing calculations, comparing values with each other, calling functions, manipulating the stack, implementing loops and conditions, etc.

For example:

	mov  N, 8

	add  3, 4

	push 9

	call io.print, "Hello World!"

You can find a complete listing of all supported instructions in the Instruction Reference.

### Concatenation

Instructions can be concatenated on one line with the pipe character `|`.

For example:

	add | mul | sub | div

	cmp A, 5 | jle 2f

	call list.filter, {cmp _,10 | setle}, [1, 2, 3, 4]

## Stack

The stack is the central data structure of the language. All instructions transform the stack in one way or another.

All arguments of an instruction are automatically pushed onto the stack. The instruction then pops the required amount of elements from the stack (consume), computes the result, and pushes it to the stack (produce).

The following example shows how the stack changes when the instructions are executed:

	push 6, 2  ;[6, 2]
	add  8, 4  ;[12, 6, 2]
	mul  2     ;[24, 6, 2]
	sub        ;[18, 2]
	div        ;[9]

Notice how `sub` has no arguments. It consumes two values from the stack and produces one.

### Stack Manipulation

There are a few instructions that are specifically designed to manipulate the stack in different ways. The most prominent ones are:

    dup   duplicate the first value from the top
          [3] -> [3, 3]

    over  duplicate the second value from the top
          [3, 4] -> [4, 3, 4]

    rot   rotate three elements at the top
          [1, 2, 3] -> [2, 3, 1]

    swap  swap the two elements at the top
          [3, 4] -> [4, 3]

### Stack Character

The stack character `_` can be used to place a value from the stack into the arguments list.

	push 1, 2  ;[1, 2]
	push _, 3  ;[1, 3, 2]

This is useful for instructions where the position of the value matters for the computation, like subtractions and comparisons.

	          ;[4]
	cmp 5     ;[] CMP=5-4=1
	setl      ;[false]

	          ;[4]
	cmp _, 5  ;[] CMP=4-5=-1
	setl      ;[true]

## Variables

A Variable is a symbolic reference to an object. Any value can be assigned to a variable. To assign a value to a variable use the `mov` instruction.

	mov A, 3  ;A=3

Once a value is assigned to a variable, you can use it in other instructions:

	add A, 5  ;A=8

You can also assign values to variables with the `pop` instruction:

	push 1, 2, 3  ;[1, 2, 3]
	pop  A, B     ;[3] A=1, B=2

### Visibility of Variables

The language has the concept of a function scope, similar to other languages that have functions.

Variables are local to the function scope they are defined in. When you call a function, a new function scope is created. When you return from a function, its function scope is deleted.

Variables are not affected by jump instructions.

## Labels

A label is a symbolic reference to an instruction address. Labels are local to the module they are defined in. Labels are objects like numbers and strings, so you can treat them as values.

There are three types of labels:

1. Public symbolic labels
2. Private symbolic labels
3. Private numeric labels

To define a public symbolic label, you write a name, followed by a colon `:`.

	label:

The name must be unique within the module. You can reference the label from within the same module like this:

	label

To reference a public label from another module, you must preface the label with the name of the module, followed by a period:

	module.label

To define a private label, preface the name with a period `.`.

	.local_label:

The name must be unique within the module. You can reference the label from within the same module like this:

	.local_label

You cannot reference a private label from another module.

To define a private numeric label, write a positive integer number followed by a colon `:`:

	1:
	2:
	1:
	33:

Numeric labels can be redefined in any order. You reference the label from within the module by appending the character `b` or `f`, to indicate where the label is located relative to the reference:

	42b  ;a backward reference to the numeric label 42
	42f  ;a forward reference to the numeric label 42

## Branching <a name="sec_branching"></a>

If you want to jump to a label in your code, use the `jmp` instruction.

	jmp label

If the jump to a label depends on a condition, use a conditional jump.

	cmp 3, 8
	jl  labelA  ;if 3 < 8 jump to labelA
	jmp labelB  ;else     jump to labelB

This can also be used to create loops:

	    mov  N, 8
	0:  cmp  N, 0
	    je   1f
	    call io.print, "i:", N
	    dec  N
	    jmp  0b
	1:  call io.print, "done!"

There is a conditional jump instruction for every case that you might need after a comparison. Check the Instruction Reference for more details.

## Functions

To call a function, you use the `call` instruction. To return from a function, you use the `ret` instruction.

	sum3:                   ;[1, 2, 3]
	    add                 ;[3, 3]
	    add                 ;[6]
	    ret                 ;[6]

	main:
	    call sum3, 1, 2, 3  ;[6]
	    ret

If you use variables, you have to be a bit more expressive:

	sum3:                   ;[1, 2, 3]
	    pop A, B, C         ;[] A=1, B=2, C=3
	    mov r, 0            ;[] r=0
	    add r, A, B, C      ;[] r=6
	    ret r               ;[6]
	
	main:
	    call sum3, 1, 2, 3  ;[6]
	    ret

You can also use labels as instructions to perform function calls:

	main:
	    sum3 1, 2, 3  ;[6]
	    ret

Whenever the interpreter encounters a label as the instruction name, it performs a function call by default.

### Deferring Function Calls

Using the `defer` instruction, you can defer the call of a function until the end of the function scope.

	init:
	    defer io.print, 0
	    defer io.print, 1
	    defer io.print, 2

	    print "A"
	    print "B"
	    print "C"
	    ret

Will print the following to the console:

	A
	B
	C
	2
	1
	0

Note that the numbers are printed in reverse.

This is useful for situations where something needs to be done right before the function is exited.

## Higher Order Functions

Since labels are objects, you can push a label to the stack, or assign it to a variable, and call it at a later time:

	mov  fn, label
	call fn

You can also pass labels to other functions:

	call list.reduce, math.add, [1, 2, 3]  ;[6]

This is similar to function pointers in other languages.

## Anonymous Functions

You can create an anonymous function by placing concatenated instructions between two curly braces `{`, `}`.

	{ add | sub 3 | call fn | dup }

You can write the previous example like this:

	call list.reduce, { add }, [1, 2, 3]  ;[6]

Anonymous functions are objects that you can push, pop, assign to variables, and call:

	push { add A, B }
	[...]
	call

Variables used in anonymous functions will not be resolved until the function is called!

_Note: The curly braces only tell the interpreter that this block of text is an object, and that it does not have to be interpreted, unless execute is called on the object._

## Modules

A module is a collection of instructions and labels. Every file or document in SASM is a module. The name of the module is the file name, without the `.sasm` extension. You can only call public symbolic labels from other modules.

The language implementation comes with a few core modules, which offer functions for stack, list, and string manipulations. You can find a more detailed listing in the Core Reference.

# Instruction Reference

## Stack

pop - pop one or more values from stack

	pop         ;pop()
	pop N       ;N = pop()
	pop N,M     ;N = pop(), M = pop()
	pop _,M     ;pop(), M = pop()

push - push one or more values to the stack

	push N      ;push(N)
	push N,M    ;push(N), push(M)

## Assignment

mov - assign to variable

	mov N       ;N = pop()
	mov N,M     ;N = M

## Arithmetic

add - add two or more numbers

	add         ;push(pop() + pop())
	add N       ;N = N + pop()
	add N,M     ;N = N + M

sub - subtract two or more numbers

	sub         ;push(pop() - pop())
	sub N       ;N = N - pop()
	sub N,M     ;N = N - M

mul - multiply two or more numbers

	mul         ;push(pop() * pop())
	mul N       ;N = N * pop()
	mul N,M     ;N = N * M

div - divide two numbers

	div         ;push(pop() / pop())
	div N       ;N = N / pop()
	div N,M     ;N = N / M

mod - mod

	mod         ;push(pop() % pop())
	mod N       ;N = N % pop()
	mod N,M     ;N = N % M

inc - increment one or more numbers

	inc         ;push(pop() + 1)
	inc N       ;N = N + 1
	inc N,M     ;N = N + 1, M = M + 1

dec - decrement one or more numbers

	dec         ;push(pop() - 1)
	dec N       ;N = N - 1
	dec N,M     ;N = N - 1, M = M - 1

## Logic

All logic instructions work with boolean values.

and

	and         ;push(pop() & pop())
	and N       ;N = N & pop()
	and N,M     ;N = N & M

or

	or          ;push(pop() | pop())
	or N        ;N = N | pop()
	or N,M      ;N = N | M

xor

	xor         ;push(pop() ^ pop())
	xor N       ;N = N ^ pop()
	xor N,M     ;N = N ^ M

not

	not         ;push(!pop())
	not N       ;N = !N

## Comparison

cmp - compares two values and sets the comparison flag

	cmp         ;cmp(pop(), pop())
	cmp N       ;cmp(N, pop())
	cmp N,M     ;cmp(N, M)

The comparison must be followed by a conditional jump, or a set instruction.

The set instructions evaluate the comparison flag and either push a boolean value to the stack, or assign a boolean value to a variable.

sete

	sete        ;push(CMP == 0)
	sete N      ;N = (CMP == 0)

setne

	setne       ;push(CMP != 0)
	setne N     ;N = (CMP != 0)

setg

	setg        ;push(CMP > 0)
	setg N      ;N = (CMP > 0)

setge

	setge       ;push(CMP >= 0)
	setge N     ;N = (CMP >= 0)

setl

	setl        ;push(CMP < 0)
	setl N      ;N = (CMP < 0)

setle

	setle       ;push(CMP <= 0)
	setle N     ;N = (CMP <= 0)

## Branching

call - call function

	call L,N,M  ;push(M), push(N), jump to label

ret - return from function

	ret
	ret N       ;push(N)
	ret N,M     ;push(M), push(N)

defer - defer function call to end of function scope

	defer L,N,M

jmp - jump to label

	jmp L

je - jump to label if equal

	je L        ;(CMP == 0)

jg - jump to label if greater

	jg L        ;(CMP > 0)

jge - jump to label if greater or equal

	jge L       ;(CMP >= 0)

jl - jump to label if less

	jl L        ;(CMP < 0)

jle - jump to label if less or equal

	jle L       ;(CMP <= 0)

jne - jump to label if not equal

	jne L       ;(CMP != 0)

jng - jump to label if not greater

	jng L       ;!(CMP > 0)

jnge - jump to label if not greater or equal

	jnge L       ;!(CMP >= 0)

jnl - jump to label if not less

	jnl L        ;!(CMP < 0)

jnle - jump to label if not less or equal

	jnle L       ;!(CMP <= 0)

# Core Reference

## Bool

bool.and
bool.not
bool.or
bool.xor

## IO

io.print
io.readlines
io.writefile

## List

list.append
list.assoc
list.assocs
list.create
list.filter
list.foreach
list.get
list.group
list.head
list.indexof
list.init
list.isempty
list.last
list.map
list.merge
list.pack
list.reduce
list.reverse
list.rotate
list.size
list.sort
list.split ???
list.sublist
list.tail
list.unpack
list.zip
list.zipwith

## Math

math.add
math.sub
math.mul
math.div
math.mod

## Stack

stack.drop
stack.dup
stack.over
stack.rot
stack.size
stack.swap

## String

str.concat
str.explode
str.len
str.split
str.substr
str.tolower
str.toupper

## System

sys.args
sys.exec
sys.exit
sys.in
sys.err
sys.out

## Type

type.isbool
type.isint
type.islabel
type.islist
type.issame
type.isstr
type.isblock
type.toint
type.tostring
